<?php
require '../config.php';
session_start();
require 'navbar.php';

if (!isset($_SESSION['user_id'])) {
  header("Location: ../login.php");
  exit;
}

$user_id = $_SESSION['user_id'];

// Check for payment success notification
$payment_success = false;
$payment_info = null;
if (isset($_GET['payment_success']) && $_GET['payment_success'] == '1' && isset($_SESSION['payment_success'])) {
  $payment_success = true;
  $payment_info = $_SESSION['payment_success'];
  unset($_SESSION['payment_success']);
}

// Get filter parameters
$tanggal_dari = $_GET['tanggal_dari'] ?? '';
$tanggal_sampai = $_GET['tanggal_sampai'] ?? '';
$status = isset($_GET['status']) ? $_GET['status'] : '';

// Status configuration
$all_statuses = [
  '' => '-- Semua --',
  'menunggu' => 'Menunggu Konfirmasi',
  'dibayar' => 'Sudah Dibayar',
  'dikirim' => 'Sedang Dikirim',
  'selesai' => 'Pesanan Selesai',
  'dibatalkan' => 'Dibatalkan'
];

// Get status counts
$status_list = ['menunggu', 'dibayar', 'dikirim', 'selesai', 'dibatalkan'];
$status_counts = [];

foreach ($status_list as $status_item) {
  $stmt = $conn->prepare("SELECT COUNT(*) as jumlah FROM transaksi WHERE pengguna_id = ? AND status = ?");
  if ($stmt) {
    $stmt->bind_param("is", $user_id, $status_item);
    $stmt->execute();
    $result_status = $stmt->get_result();
    if ($result_status) {
      $row_status = $result_status->fetch_assoc();
      $status_counts[$status_item] = $row_status['jumlah'];
    } else {
      $status_counts[$status_item] = 0;
    }
  } else {
    $status_counts[$status_item] = 0;
  }
}

$status_config = [
  'menunggu' => ['color' => 'warning', 'icon' => 'clock', 'text' => 'Menunggu Konfirmasi'],
  'dibayar' => ['color' => 'success', 'icon' => 'check-circle', 'text' => 'Sudah Dibayar'],
  'dikirim' => ['color' => 'info', 'icon' => 'truck', 'text' => 'Sedang Dikirim'],
  'selesai' => ['color' => 'primary', 'icon' => 'check-double', 'text' => 'Pesanan Selesai'],
  'dibatalkan' => ['color' => 'danger', 'icon' => 'times-circle', 'text' => 'Dibatalkan']
];

// Build WHERE clause
$where_clauses = ["pengguna_id = ?"];
$params = [$user_id];
$param_types = "i";

if ($tanggal_dari && $tanggal_sampai) {
  $where_clauses[] = "DATE(dibuat_pada) BETWEEN ? AND ?";
  $params[] = $tanggal_dari;
  $params[] = $tanggal_sampai;
  $param_types .= "ss";
} elseif ($tanggal_dari) {
  $where_clauses[] = "DATE(dibuat_pada) >= ?";
  $params[] = $tanggal_dari;
  $param_types .= "s";
} elseif ($tanggal_sampai) {
  $where_clauses[] = "DATE(dibuat_pada) <= ?";
  $params[] = $tanggal_sampai;
  $param_types .= "s";
}

if ($status !== '' && array_key_exists($status, $all_statuses)) {
  $where_clauses[] = "status = ?";
  $params[] = $status;
  $param_types .= "s";
}

$query_where = implode(" AND ", $where_clauses);
$query = "SELECT * FROM transaksi WHERE $query_where ORDER BY dibuat_pada DESC";

$stmt = $conn->prepare($query);
if ($stmt === false) {
  die("Prepare failed: " . $conn->error);
}

$stmt->bind_param($param_types, ...$params);
if (!$stmt->execute()) {
  die("Execute failed: " . $stmt->error);
}
$result = $stmt->get_result();

function formatKodeTransaksi($id, $timestamp) {
  $tanggal = date('Ymd', strtotime($timestamp));
  return 'TRX-' . $tanggal . '-' . str_pad($id, 4, '0', STR_PAD_LEFT);
}
?>

<!DOCTYPE html>
<html lang="id">

<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Riwayat Pemesanan - vedge</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://fonts.googleapis.com/css2?family=Playfair+Display:wght@400;500;600;700;800;900&display=swap" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css" rel="stylesheet">
  <style>
    body {
      background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
      color: #1a1a1a;
      font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      line-height: 1.6;
    }

    .hero-section {
      background: white;
      padding: 40px 0;
      margin-bottom: 40px;
      border-bottom: 1px solid #e9ecef;
    }

    .hero-title {
      font-family: 'Playfair Display', serif;
      font-size: 2.5rem;
      font-weight: 700;
      color: #1a1a1a;
      margin-bottom: 10px;
    }

    .hero-subtitle {
      color: #666;
      font-size: 1.1rem;
    }

    .main-container {
      background: white;
      border-radius: 20px;
      box-shadow: 0 10px 40px rgba(0, 0, 0, 0.08);
      overflow: hidden;
      margin-bottom: 40px;
    }

    .status-card {
      background: white;
      border: 1px solid #e9ecef;
      border-radius: 15px;
      padding: 20px;
      text-align: center;
      transition: all 0.3s ease;
      height: 100%;
    }

    .status-card:hover {
      transform: translateY(-2px);
      box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
    }

    .status-icon {
      font-size: 2rem;
      margin-bottom: 10px;
    }

    .status-count {
      font-size: 1.5rem;
      font-weight: 700;
      margin-bottom: 5px;
    }

    .filter-section {
      background: #f8f9fa;
      padding: 25px;
      border-bottom: 1px solid #e9ecef;
    }

    .filter-title {
      font-family: 'Playfair Display', serif;
      font-size: 1.3rem;
      font-weight: 600;
      color: #1a1a1a;
      margin-bottom: 20px;
    }

    .table-container {
      padding: 30px;
    }

    .table {
      background: white;
      border-radius: 15px;
      overflow: hidden;
      box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
    }

    .table thead th {
      background: #f8f9fa;
      color: #1a1a1a;
      font-weight: 600;
      padding: 15px;
      border: none;
      font-size: 0.85rem;
      text-transform: uppercase;
      letter-spacing: 0.5px;
    }

    .table tbody tr {
      border-bottom: 1px solid #f0f0f0;
      transition: all 0.3s ease;
    }

    .table tbody tr:hover {
      background: #f8f9fa;
    }

    .table td {
      padding: 15px;
      vertical-align: middle;
      border: none;
    }

    .status-badge {
      padding: 6px 12px;
      border-radius: 20px;
      font-size: 0.75rem;
      font-weight: 600;
      text-transform: uppercase;
      letter-spacing: 0.5px;
    }

    .status-menunggu { background: #fff3cd; color: #856404; }
    .status-dibayar { background: #d1e7dd; color: #0f5132; }
    .status-dikirim { background: #cff4fc; color: #055160; }
    .status-selesai { background: #cfe2ff; color: #084298; }
    .status-dibatalkan { background: #f8d7da; color: #721c24; }

    .action-btn {
      padding: 6px 12px;
      border-radius: 8px;
      font-size: 0.8rem;
      font-weight: 600;
      text-decoration: none;
      transition: all 0.3s ease;
      display: inline-flex;
      align-items: center;
      gap: 5px;
    }

    .action-btn:hover {
      transform: translateY(-1px);
      text-decoration: none;
    }

    .empty-state {
      text-align: center;
      padding: 60px 30px;
    }

    .empty-icon {
      font-size: 4rem;
      color: #ccc;
      margin-bottom: 20px;
    }

    .empty-title {
      font-family: 'Playfair Display', serif;
      font-size: 1.5rem;
      color: #1a1a1a;
      margin-bottom: 10px;
    }

    .empty-text {
      color: #666;
      margin-bottom: 30px;
    }

    .payment-timer {
      background: rgba(220, 53, 69, 0.1);
      padding: 4px 8px;
      border-radius: 6px;
      font-size: 0.75rem;
      color: #dc3545;
      font-weight: 600;
    }

    @media (max-width: 768px) {
      .hero-title {
        font-size: 2rem;
      }

      .main-container {
        margin: 20px;
        border-radius: 15px;
      }

      .table-container {
        padding: 20px;
      }

      .status-card {
        margin-bottom: 15px;
      }
    }

    @media (max-width: 480px) {
      .hero-title {
        font-size: 1.8rem;
      }

      .main-container {
        margin: 15px;
      }

      .table-responsive {
        font-size: 0.85rem;
      }

      .action-btn {
        padding: 4px 8px;
        font-size: 0.75rem;
      }
    }
  </style>
</head>

<body>
  <!-- Hero Section -->
  <section class="hero-section">
    <div class="container">
      <div class="text-center">
        <h1 class="hero-title">
          <i class="fas fa-history me-3"></i>
          Riwayat Pemesanan
        </h1>
        <p class="hero-subtitle">Kelola dan pantau semua pesanan Anda</p>
      </div>
    </div>
  </section>

  <div class="container">
    <!-- Success/Error Messages -->
    <?php if (isset($_SESSION['success_message'])): ?>
      <div class="alert alert-success alert-dismissible fade show" role="alert">
        <i class="fas fa-check-circle me-2"></i><?php echo $_SESSION['success_message']; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
      </div>
      <?php unset($_SESSION['success_message']); ?>
    <?php endif; ?>

    <?php if (isset($_SESSION['error_message'])): ?>
      <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <i class="fas fa-exclamation-circle me-2"></i><?php echo $_SESSION['error_message']; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
      </div>
      <?php unset($_SESSION['error_message']); ?>
    <?php endif; ?>

    <?php if ($payment_success && $payment_info): ?>
      <div class="alert alert-success alert-dismissible fade show" role="alert">
        <div class="row align-items-center">
          <div class="col-auto">
            <div class="bg-success text-white rounded-circle d-flex align-items-center justify-content-center" style="width: 50px; height: 50px;">
              <i class="fas fa-check"></i>
            </div>
          </div>
          <div class="col">
            <h6 class="fw-bold mb-2">
              <i class="fas fa-check-circle me-2"></i>Pembayaran Berhasil Dikonfirmasi!
            </h6>
            <p class="mb-0">Bukti pembayaran Anda telah diterima dan sedang diproses.</p>
          </div>
        </div>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
      </div>
    <?php endif; ?>



    <!-- Filter Section -->
    <div class="main-container">
      <div class="filter-section">
        <h3 class="filter-title">
          <i class="fas fa-filter me-2"></i>
          Filter Pesanan
        </h3>
        <form method="GET" class="row g-3">
          <div class="col-md-3">
            <label for="tanggal_dari" class="form-label">Dari Tanggal</label>
            <input type="date" id="tanggal_dari" name="tanggal_dari" class="form-control" value="<?= htmlspecialchars($tanggal_dari) ?>">
          </div>
          <div class="col-md-3">
            <label for="tanggal_sampai" class="form-label">Sampai Tanggal</label>
            <input type="date" id="tanggal_sampai" name="tanggal_sampai" class="form-control" value="<?= htmlspecialchars($tanggal_sampai) ?>">
          </div>
          <div class="col-md-3">
            <label for="status" class="form-label">Status</label>
            <select name="status" id="status" class="form-select">
              <?php foreach ($all_statuses as $key => $value): ?>
                <option value="<?= $key ?>" <?= ($status === $key) ? 'selected' : '' ?>><?= $value ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-3 d-flex align-items-end">
            <div class="d-flex gap-2 w-100">
              <button type="submit" class="btn btn-dark flex-fill">
                <i class="fas fa-filter me-2"></i>Filter
              </button>
              <a href="riwayat.php" class="btn btn-outline-secondary">
                <i class="fas fa-sync-alt"></i>
              </a>
            </div>
          </div>
        </form>
      </div>

      <!-- Transactions Table -->
      <div class="table-container">
        <div class="table-responsive">
          <table class="table">
            <thead>
              <tr>
                <th>ID Transaksi</th>
                <th>Total</th>
                <th>Status</th>
                <th>Resi</th>
                <th>Bukti Pembayaran</th>
                <th>Tanggal</th>
                <th>Aksi</th>
              </tr>
            </thead>
            <tbody>
              <?php if ($result->num_rows > 0): ?>
                <?php while ($row = $result->fetch_assoc()): ?>
                  <tr>
                    <td>
                      <strong><?= htmlspecialchars(formatKodeTransaksi($row['id'], $row['dibuat_pada'])) ?></strong>
                    </td>
                    <td>
                      <span class="fw-bold text-primary">Rp<?= number_format($row['total_harga'], 0, ',', '.') ?></span>
                    </td>
                    <td>
                      <span class="status-badge status-<?= htmlspecialchars($row['status']) ?>">
                        <?= htmlspecialchars($all_statuses[$row['status']] ?? ucfirst($row['status'])) ?>
                      </span>
                    </td>
                    <td>
                      <?= $row['resi'] ? htmlspecialchars($row['resi']) : '<span class="text-muted">-</span>' ?>
                    </td>
                    <td>
                      <?php if ($row['bukti_pembayaran']): ?>
                        <a href="../upload/<?= htmlspecialchars($row['bukti_pembayaran']) ?>" target="_blank" class="action-btn btn btn-outline-primary btn-sm">
                          <i class="fas fa-eye"></i>Lihat
                        </a>
                      <?php else: ?>
                        <span class="text-muted">Belum Ada</span>
                      <?php endif; ?>
                    </td>
                    <td><?= date('d M Y H:i', strtotime($row['dibuat_pada'])) ?></td>
                    <td>
                      <div class="d-flex flex-column gap-1">
                        <?php if ($row['status'] === 'menunggu' && !$row['bukti_pembayaran']): ?>
                          <?php
                          $created_time = strtotime($row['dibuat_pada']);
                          $current_time = time();
                          $elapsed_seconds = $current_time - $created_time;
                          $time_limit = 5 * 60;
                          $remaining_seconds = $time_limit - $elapsed_seconds;

                          if ($remaining_seconds > 0):
                            $minutes = 5;
                            $seconds = 0;
                          ?>
                            <div class="payment-timer mb-2">
                              <i class="fas fa-clock me-1"></i>
                              <span id="timer-<?= $row['id'] ?>"><?= sprintf('%02d:%02d', $minutes, $seconds) ?></span>
                            </div>
                            <a href="upload_bukti.php?transaksi_id=<?= $row['id'] ?>" class="action-btn btn btn-warning btn-sm">
                              <i class="fas fa-upload"></i>Upload Bukti
                            </a>
                          <?php else: ?>
                            <span class="text-danger small">Waktu habis</span>
                          <?php endif; ?>
                        <?php elseif ($row['status'] === 'dikirim'): ?>
                          <button class="action-btn btn btn-success btn-sm" onclick="confirmSelesai(<?= $row['id'] ?>)" title="Konfirmasi pesanan sudah diterima">
                            <i class="fas fa-check"></i>Pesanan Selesai
                          </button>
                        <?php elseif ($row['status'] === 'selesai'): ?>
                          <button class="action-btn btn btn-warning btn-sm" onclick="openReviewModal(<?= $row['id'] ?>)" title="Beri ulasan untuk pesanan ini">
                            <i class="fas fa-star"></i>Beri Ulasan
                          </button>
                        <?php endif; ?>
                        <a href="detail_transaksi.php?id=<?= htmlspecialchars($row['id']) ?>" class="action-btn btn btn-outline-dark btn-sm">
                          <i class="fas fa-eye"></i>Detail
                        </a>
                      </div>
                    </td>
                  </tr>
                <?php endwhile; ?>
              <?php else: ?>
                <tr>
                  <td colspan="7">
                    <div class="empty-state">
                      <div class="empty-icon">
                        <i class="fas fa-box-open"></i>
                      </div>
                      <h4 class="empty-title">Tidak Ada Transaksi Ditemukan</h4>
                      <p class="empty-text">Coba ubah kriteria filter Anda atau buat pesanan baru.</p>
                      <a href="../index.php" class="btn btn-dark">
                        <i class="fas fa-shopping-bag me-2"></i>Mulai Berbelanja
                      </a>
                    </div>
                  </td>
                </tr>
              <?php endif; ?>
            </tbody>
          </table>
        </div>
      </div>
    </div>
  </div>

  <!-- Upload Bukti Modal -->
  <div class="modal fade" id="uploadBuktiModal" tabindex="-1">
    <div class="modal-dialog">
      <div class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title">Upload Bukti Pembayaran</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
        </div>
        <form id="uploadBuktiForm" action="upload_bukti.php" method="POST" enctype="multipart/form-data">
          <div class="modal-body">
            <input type="hidden" name="transaksi_id" id="transaksi_id">
            <div class="mb-3">
              <label for="bukti_pembayaran" class="form-label">Pilih File Bukti Pembayaran</label>
              <input type="file" class="form-control" id="bukti_pembayaran" name="bukti_pembayaran" accept="image/*" required>
              <div class="form-text">Format: JPG, PNG, JPEG. Maksimal 5MB.</div>
            </div>
            <div class="alert alert-info">
              <i class="fas fa-info-circle"></i>
              Pastikan bukti pembayaran jelas dan dapat dibaca dengan baik.
            </div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
            <button type="submit" class="btn btn-primary">Upload Bukti</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- Review Modal -->
  <div class="modal fade" id="reviewModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
      <div class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title">Beri Ulasan Produk</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
        </div>
        <div class="modal-body" id="reviewModalBody">
          <!-- Content will be loaded dynamically -->
        </div>
      </div>
    </div>
  </div>

  <?php include '../footer.php'; ?>
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
  <script src="../assets/auto-cancel.js"></script>

  <script>
    // Upload bukti function
    function uploadBukti(transaksiId) {
      document.getElementById('transaksi_id').value = transaksiId;
      const modal = new bootstrap.Modal(document.getElementById('uploadBuktiModal'));
      modal.show();
    }

    // Open review modal function
    function openReviewModal(transaksiId) {
      document.getElementById('reviewModalBody').innerHTML = '<div class="text-center p-4"><div class="spinner-border" role="status"></div><div class="mt-2">Memuat produk...</div></div>';
      
      const modal = new bootstrap.Modal(document.getElementById('reviewModal'));
      modal.show();

      fetch('get_transaction_products.php?transaksi_id=' + transaksiId)
        .then(response => response.text())
        .then(html => {
          document.getElementById('reviewModalBody').innerHTML = html;
        })
        .catch(error => {
          document.getElementById('reviewModalBody').innerHTML = '<div class="alert alert-danger">Error memuat produk: ' + error.message + '</div>';
        });
    }

    // Global submitBulkReview function
    window.submitBulkReview = function() {
      const form = document.getElementById('transactionReviewForm');
      if (!form) {
        alert('Form tidak ditemukan. Silakan refresh halaman.');
        return;
      }

      const ratings = form.querySelectorAll('input[type="radio"]:checked');
      const textareas = form.querySelectorAll('textarea[required]');
      let isValid = true;

      const ratingGroups = form.querySelectorAll('input[type="radio"][required]');
      const uniqueGroups = new Set();
      ratingGroups.forEach(input => uniqueGroups.add(input.name));

      if (ratings.length < uniqueGroups.size) {
        alert('Harap berikan rating untuk semua produk.');
        return;
      }

      textareas.forEach(textarea => {
        if (!textarea.value.trim()) {
          isValid = false;
        }
      });

      if (!isValid) {
        alert('Harap isi semua field ulasan.');
        return;
      }

      const submitBtn = document.getElementById('submitReviewBtn');
      const originalText = submitBtn.innerHTML;

      submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Mengirim...';
      submitBtn.disabled = true;

      const formData = new FormData(form);

      fetch('proses/proses_tambah_ulasan_bulk.php', {
        method: 'POST',
        credentials: 'include',
        headers: {
          'X-Requested-With': 'XMLHttpRequest'
        },
        body: formData
      })
      .then(response => response.json())
      .then(data => {
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;

        if (data.success) {
          alert('Semua ulasan berhasil dikirim! Terima kasih atas feedback Anda.');
          const modal = bootstrap.Modal.getInstance(document.querySelector('#reviewModal'));
          if (modal) modal.hide();
          setTimeout(() => window.location.reload(), 1000);
        } else {
          alert('Error: ' + (data.message || 'Gagal mengirim ulasan. Silakan coba lagi.'));
        }
      })
      .catch(error => {
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
        alert('Terjadi kesalahan saat mengirim ulasan: ' + error.message);
      });
    };

    // Confirm selesai function
    function confirmSelesai(transaksi_id) {
      if (confirm('Apakah Anda yakin pesanan sudah diterima dan ingin menandai sebagai selesai?')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = '../proses/proses_selesai_pesanan.php';

        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'transaksi_id';
        input.value = transaksi_id;

        form.appendChild(input);
        document.body.appendChild(form);
        form.submit();
      }
    }

    // Timer countdown function
    function startTimers() {
      const timerElements = document.querySelectorAll('[id^="timer-"]');

      timerElements.forEach(function(element) {
        const timerText = element.textContent.trim();
        if (!timerText) return;

        const parts = timerText.split(':');
        if (parts.length !== 2) return;

        let totalSeconds = parseInt(parts[0]) * 60 + parseInt(parts[1]);
        if (isNaN(totalSeconds) || totalSeconds <= 0) return;

        const interval = setInterval(function() {
          totalSeconds--;

          if (totalSeconds <= 0) {
            clearInterval(interval);
            element.textContent = '00:00';
            setTimeout(() => location.reload(), 2000);
            return;
          }

          const minutes = Math.floor(totalSeconds / 60);
          const seconds = totalSeconds % 60;
          element.textContent = String(minutes).padStart(2, '0') + ':' + String(seconds).padStart(2, '0');
        }, 1000);
      });
    }

    // Initialize on page load
    document.addEventListener('DOMContentLoaded', function() {
      startTimers();
    });
  </script>
</body>

</html>